function fluorescence = analyzeVideo(videoFileName, labels)
% This function takes a grayscale video as input and produces a graph of
% how an ROI's intensity changes over time.  After providing the input
% video, the user is asked to select one background ROI and one or more
% foreground ROIs for analysis from any frame of the video.

% The program works as follows:
% (1) Ask the user for ROIs, with a magnified, scrollable, flippable heat
% map.  Note that the ROI's size will never change, but it may be
% translated in space depending on what happens to its underlying objects.
% (2) Identify the objects that make up the ROI
% (3) For each frame before and after the frame on which the ROI was drawn,
%    follow the objects that were in the ROI (based on spatial overlap).  
%    Several things can happen to the objects that will automatically be 
%    accounted for by the ROI:
%    (a) The object translates in space.  If this happens, translate the ROI
%    with the same translation vector.  The vector is determined by the
%    movement of the object's centroid from one frame to the next.
%    (b) The object scales in space (increases or decreases).  There are 2
%    kinds of scaling:
%        (i) Scaling that maintains the object's centroid location.  This is
%        a uniform expansion or contraction, and in this case the ROI is
%        not moved.
%        (ii) Scaling that shifts the object's centroid location.  This
%        happens if only part of the object expands or contracts.  In this
%        case, the ROI will be translated by the object's centroid vector
%        shift.
%    (c) The object collides with another object.  In this case, create
%    artifical object boundaries by using the object boundaries from the
%    previous frame where the objects were not touching.  Deal with the ROI
%    as above.
%    (d) The object separates into two or more objects.  Take the centroid
%    of both objects together and use that for shifting the ROI.

% This function can be run on its own or as part of the imaging recording
% package, which also controls the Polychrome light source and Andor EM
% camera.

global motionCompensation;
global vreader;
global maskOverTime;
global overlapRPOverTime;
global overlapCentroidOverTime;
global overlapAreaOverTime;
global maskdx;
global maskdy;
global firstFrameNum;
global thresh;
global fudgeFactor;  % fudge factors used for each ROI to individuate the underlying object

% PARAMETERS
debug = true;  % Whether to generate the video with motion compensation or not
motionCompensation = true;
lw = 2;  % LineWidth

thresh = -1;
fudgeFactor = -1;

vreader = VideoReader(videoFileName);
[p vidName ext] = fileparts(videoFileName);
fr = vreader.FrameRate;
h = layout(vreader);
waitfor(h, 'Pointer', 'watch');

tic;

handles = guidata(h);

% Begin image analysis and ROI object following:
% For each ROI, first start with the frame on which is was last specified.
% Analyze that frame for objects, and identify the objects that overlap
% with the ROI.  Next, for each frame before, analyze it for objects,
% identify those that were in the ROI in the primary frame, and analyze
% their shift in space.  Shift the ROI the same amount.  Continue for all
% frames before, and again for all frames after.  Do this for each ROI,
% resulting in a numFrames x numROI matrix of mean intensity values.
% Finally, subtract the values for the background ROI from all frames, and
% then divide by the initial ROI from the first frame to yield a numFrames x
% numROI matrix of change in fluorescence.  Save and plot this.
numFrames = get(vreader, 'numberOfFrames');
numROI = length(handles.roisMask);
vidWidth = vreader.Width;
vidHeight = vreader.Height;

fluorescence = zeros(numFrames, numROI);
fluorescenceWithMotion = zeros(numFrames, numROI);
fluorescenceBG = zeros(numFrames, 1);

if (isfield(handles, 'roibgMask'))
    for i = 1:numFrames
        curFrame = read(vreader, i);
        fluorescenceBG(i) = sum(curFrame(:,:,1) .* uint8(handles.roibgMask)) / sum(handles.roibgMask);
    end
end

for i = 1:numROI
    thresh = -1;  % Reset for each ROI
    fudgeFactor = -1;
        
    firstFrameNum = handles.roisFrame(i);
    curFrame = read(vreader, firstFrameNum);
    overlapRPOverTime = cell(numFrames, 1);
    overlapCentroidOverTime = zeros(numFrames, 2);
    overlapAreaOverTime = zeros(numFrames, 1);
    maskOverTime = logical(zeros(vidHeight, vidWidth, numFrames, 2));
    maskOverTime(:,:,firstFrameNum, 1) = handles.roisMask{i};
    maskOverTime(:,:,firstFrameNum, 2) = handles.roisMask{i};

    fluorescence(firstFrameNum, i) = sum(curFrame(:,:,1) .* uint8(maskOverTime(:,:,firstFrameNum,1))) / ...
                                     sum(maskOverTime(:,:,firstFrameNum,1));
    fluorescenceWithMotion(firstFrameNum, i) = fluorescence(firstFrameNum, i);
    
    % First, find the objects that overlap in the current ROI frame
    curFrameLab = findObjects(curFrame(:,:,1), maskOverTime(:,:,firstFrameNum,1));
    overlapLab = curFrameLab .* maskOverTime(:,:,firstFrameNum,1);
    overlapLabNumbers = unique(overlapLab(overlapLab > 0));
    rp = regionprops(curFrameLab, 'PixelList', 'PixelIdxList', 'Area');
    for j = 1:length(overlapLabNumbers)
        overlapRPOverTime{firstFrameNum}(end+1) = rp(overlapLabNumbers(j));
        overlapAreaOverTime(firstFrameNum) = overlapAreaOverTime(firstFrameNum) + rp(overlapLabNumbers(j)).Area;
    end
    overlapCentroidOverTime(firstFrameNum,:) = getObjCentroidFromRPs(overlapRPOverTime{firstFrameNum}, [], []);
    
    % Second, find how the objects shift in previous frames, and shift the ROI accordingly.
    maskdx = 0;
    maskdy = 0;
    for j = firstFrameNum-1:-1:1
        if (motionCompensation)
            fluorescence(j,i) = analyzeFrame(j, j+1, true);
        end
        fluorescenceWithMotion(j,i) = analyzeFrame(j, j+1, false);
        disp(strcat('Analyzed ROI #', num2str(i), ', frame #', num2str(j)));
    end
    % Third, find how the objects shift in future frames, and shift the ROI accordingly.
    maskdx = 0;
    maskdy = 0;
    for j = firstFrameNum+1:numFrames
        if (motionCompensation)
            fluorescence(j,i) = analyzeFrame(j, j-1, true);
        end
        fluorescenceWithMotion(j,i) = analyzeFrame(j, j-1, false);
        disp(strcat('Analyzed ROI #', num2str(i), ', frame #', num2str(j)));
    end
    fluorescence(:,i) = (fluorescence(:,i) - fluorescence(1,i)) ./ ...
                        (fluorescence(1, i) - fluorescenceBG);
    fluorescenceWithMotion(:,i) = (fluorescenceWithMotion(:,i) - fluorescenceWithMotion(1,i)) ./ ...
                        (fluorescenceWithMotion(1, i) - fluorescenceBG);
    disp(strcat('Finished analysis of ROI #', num2str(i)));
    
    % Generate AVI to see how ROI compares with actual objects in video
    if (debug)
        aviOut = avifile('out.avi', 'colormap', genColormap(128), 'compression', 'none', ...
            'fps', vreader.FrameRate);
        for j = 1:numFrames
            frame = read(vreader, j);
            f = frame(:,:,1);
            c = edge(maskOverTime(:,:,j,1), 'canny');
            f(c == 1) = 255;
            aviOut = addframe(aviOut, f);
        end
        aviOut = close(aviOut);
        beep;
        keyboard;
    end
end

% Plot without motion compensation first
figure
plot((1:numFrames)/fr, fluorescenceWithMotion*100, 'LineWidth', lw);
ylabel('\DeltaF/F (%)', 'FontWeight', 'bold');
xlabel('sec', 'FontWeight', 'bold');
title(strcat([vidName ' - No Motion Compensation']), 'FontWeight', 'bold');
prettify;
legend(labels);

% Plot with motion compensation second
figure
plot((1:numFrames)/fr, fluorescence*100, 'LineWidth', lw);
ylabel('\DeltaF/F (%)', 'FontWeight', 'bold');
xlabel('sec', 'FontWeight', 'bold');
title(strcat([vidName ' - Motion Compensated']), 'FontWeight', 'bold');
prettify;
legend(labels);


toc;
beep;


function f = analyzeFrame(j, jp, compensateMotion)
    global vreader;
    global maskOverTime;
    global overlapRPOverTime;
    global overlapCentroidOverTime;
    global overlapAreaOverTime;
    global maskdx;
    global maskdy;
    global firstFrameNum;
    
    if (j == 27)
        a = 0;
    end
    
    curFrame = read(vreader, j);
    if (compensateMotion)
        curFrameLab = findObjects(curFrame(:,:,1), maskOverTime(:,:,jp,1));
        overlapLabNumbers = [];
        % First find objects that overlap with ROI - has little effect and doesn't quite make sense to do.
        % overlapLab = curFrameLab .* maskOverTime(:,:,jp);
        % overlapLabNumbers = unique(overlapLab(overlapLab > 0));
        for k = 1:length(overlapRPOverTime{jp})  % Then find objects that overlap with prev ROI objects
            overlapLabNumbers = cat(1, overlapLabNumbers, curFrameLab(overlapRPOverTime{jp}(k).PixelIdxList));
        end
        overlapLabNumbers = unique(overlapLabNumbers(overlapLabNumbers > 0));
        rp = regionprops(curFrameLab, 'PixelList', 'PixelIdxList', 'Area');
        for k = 1:length(overlapLabNumbers)
            overlapRPOverTime{j}(end+1) = rp(overlapLabNumbers(k));
            overlapAreaOverTime(j) = overlapAreaOverTime(j) + rp(overlapLabNumbers(k)).Area;
        end
        overlapCentroidOverTime(j,:) = getObjCentroidFromRPs(overlapRPOverTime{j}, overlapRPOverTime{jp}, ...
                                        overlapCentroidOverTime(jp,:), overlapAreaOverTime(jp));

        dx = overlapCentroidOverTime(j,1) - overlapCentroidOverTime(jp,1);
        dy = overlapCentroidOverTime(j,2) - overlapCentroidOverTime(jp,2);
        if (dx == 0 && dy == 0)
            overlapAreaOverTime(j) = overlapAreaOverTime(jp);
        end
        if (~isnan(dx) && ~isnan(dy))  % Happens if there are no objects in the ROI to start
            maskdx = maskdx + dx;
            maskdy = maskdy + dy;
            tform = maketform('affine', [1 0 0; 0 1 0; maskdx maskdy 1]);

            maskOverTime(:,:,j,1) = imtransform(maskOverTime(:,:,firstFrameNum,1), tform, 'bilinear', ...
                                              'XData', [1 size(maskOverTime(:,:,firstFrameNum,1),2)], ...
                                              'YData', [1 size(maskOverTime(:,:,firstFrameNum,1),1)]);
        else 
            maskOverTime(:,:,j,1) = maskOverTime(:,:,firstFrameNum,1);
        end
        f = sum(curFrame(:,:,1) .* uint8(maskOverTime(:,:,j,1))) / sum(maskOverTime(:,:,j,1));
    else
        maskOverTime(:,:,j,2) = maskOverTime(:,:,firstFrameNum,2);
        f = sum(curFrame(:,:,1) .* uint8(maskOverTime(:,:,j,2))) / sum(maskOverTime(:,:,j,2));
    end
        

function lab = findObjects(cdata, roiMask)
% This function analyzes the frame for objects.  Inspired by getLabelsInSpace from WormWatcher.
global thresh;
global fudgeFactor;

cdataNorm = single(cdata)/255;
roiVals = cdataNorm .* roiMask;
roiVals = unique(roiVals(roiVals > 0));  % List of pixel values in ROI
perimeterMask = edge(roiMask, 'canny');
perimeterVals = cdataNorm(perimeterMask == 1);
tIndex = find(roiVals == max(perimeterVals));
if (tIndex < length(roiVals))
    thresh = roiVals(find(roiVals == max(perimeterVals)) + 1);
end
lab = findLabels(cdataNorm, thresh, 1);

%{
cdataNorm = single(cdata)/255;
roiVals = cdataNorm .* roiMask;
roiVals = unique(roiVals(roiVals > 0));
%c = conv(ones(10,1), diff(roiVals));
%cind = find(c == max(c), 1);
%thresh = roiVals(cind);
thresh = roiVals(round(length(roiVals) * .65));  % Take the 70th precentile value as the threshold
%thresh = median(roiVals);
lab = findLabels(cdataNorm, thresh, 1);
%}
%{
if (thresh == -1)  % Not yet defined - first time findObjects is called for this ROI
    thresh = graythresh(cdataNorm);
end
% On the first run, find fudgefactor by iterating until objects in the ROI
% do not reach beyond.
if (fudgeFactor == -1)
    fudgeFactor = 0.1;
    inc = 0.1;
    while (true)
        lab = findLabels(cdataNorm, thresh, fudgeFactor);
        overlapLab = lab .* roiMask;
        overlapLabNumbers = unique(overlapLab(overlapLab > 0));
        roiMaskInvert = ~roiMask;
        invertOverlapLab = lab .* roiMaskInvert;
        invertOverlapLabNumbers = unique(invertOverlapLab(invertOverlapLab > 0));
        extendsBeyondROI = any(ismember(overlapLabNumbers, invertOverlapLabNumbers));
        if (extendsBeyondROI)
            fudgeFactor = fudgeFactor + inc;
        else
            break; 
        end
    end
end
lab = findLabels(cdataNorm, thresh, fudgeFactor);
%}
function lab = findLabels(cdataNorm, t, ff)
cdataThresh = cdataNorm - ff*t;
bin = im2bw(cdataThresh,0);
binClean = bwmorph(bin, 'clean');
lab = bwlabel(binClean, 8);


function p = getObjCentroidFromRPs(rps, prevrps, prevp, prevarea)
p = prevp;
area = 0;
if (~isempty(rps))
    minx = min(rps(1).PixelList(:,1));
    maxx = max(rps(1).PixelList(:,1));
    miny = min(rps(1).PixelList(:,2));
    maxy = max(rps(1).PixelList(:,2));
    area = length(rps(1).PixelList);
    for i=2:length(rps)
        minx = min(minx, min(rps(i).PixelList(:,1)));
        maxx = max(maxx, max(rps(i).PixelList(:,1)));
        miny = min(miny, min(rps(i).PixelList(:,2)));
        maxy = max(maxy, max(rps(i).PixelList(:,2)));
        area = area + length(rps(i).PixelList);
    end
    x = minx + (maxx-minx)/2;
    y = miny + (maxy-miny)/2;
    if (~isempty(prevrps) && ~isempty(prevp) && area > 2 * prevarea)  % 5 or 2?
        x = prevp(1);
        y = prevp(2);
    end
    p = [x y];
end
    
%{
x = 0;
y = 0;
area = 0;
for i = 1:length(rps)
    x = x + sum(rps(i).PixelList(:,1));
    y = y + sum(rps(i).PixelList(:,2));
    area = area + length(rps(i).PixelList);
end
% If the area suddenly expanded, it's likely that the ROI has faded to
% background, so don't move it.
if (~isempty(prevrps) && area > 2 * prevarea)  % 5 or 2?
    x = prevp(1);
    y = prevp(2);
else
    x = x / area;
    y = y / area;
end
p = [x y];
%}
function cmap = genColormap(steps)

cmap = zeros(steps, 3);
setStep = round(steps/5);
cnt = 1;
for i=1:setStep
    cmap(cnt,:) = [0 0 round(i/setStep*255)];
    cnt = cnt + 1;
end
for i=1:setStep
    cmap(cnt,:) = [0 round(i/setStep*255) 255];
    cnt = cnt + 1;
end
for i=1:setStep
    cmap(cnt,:) = [0 255 255-round(i/setStep*255)];
    cnt = cnt + 1;
end
for i=1:setStep
    cmap(cnt,:) = [round(i/setStep*255) 255 0];
    cnt = cnt + 1;
end
for i=1:setStep
    cmap(cnt,:) = [255 255-round(i/setStep*255) 0];
    cnt = cnt + 1;
end
cmap = cmap ./ 255;